<?php
abstract class API
{
    /**
     * Property: method
     * The HTTP method this request was made in, either GET, POST, PUT or DELETE
     */
    protected $method = '';
    /**
     * Property: module
     * The Module requested in the URI. eg: /files
     */
    protected $module = '';
    /**
     * Property: verb
     * An optional additional descriptor about the module, used for things that can
     * not be handled by the basic methods. eg: /files/process
     */
    protected $verb = '';
    /**
     * Property: args
     * Any additional URI components after the module and verb have been removed, in our
     * case, an integer ID for the resource. eg: /<module>/<verb>/<arg0>/<arg1>
     * or /<module>/<arg0>
     */
    protected $args = Array();
    /**
     * Property: file
     * Stores the input of the PUT request
     */
    protected $file = Null;

    /**
     * Constructor: __construct
     * Allow for CORS, assemble and pre-process the data
     */
    public function __construct($request) {
        header("Access-Control-Allow-Orgin: *");
        header("Access-Control-Allow-Methods: *");
        header("Content-Type: application/json");

		$this->file = file_get_contents("php://input");
		
        $this->args = explode('/', rtrim($request, '/'));
        $this->module = array_shift($this->args);
        if (array_key_exists(0, $this->args) && !is_numeric($this->args[0])) {
            $this->verb = array_shift($this->args);
        }

        $this->method = $_SERVER['REQUEST_METHOD'];
        if ($this->method == 'POST' && array_key_exists('HTTP_X_HTTP_METHOD', $_SERVER)) {
            if ($_SERVER['HTTP_X_HTTP_METHOD'] == 'DELETE') {
                $this->method = 'DELETE';
            } else if ($_SERVER['HTTP_X_HTTP_METHOD'] == 'PUT') {
                $this->method = 'PUT';
            } else {
                throw new Exception("Unexpected Header");
            }
        }

        switch($this->method) {
        case 'DELETE':
        case 'POST':
            $this->request = $this->_cleanInputs($_POST);
            break;
        case 'GET':
            $this->request = $this->_cleanInputs($_GET);
            break;
        case 'PUT':
            $this->request = $this->_cleanInputs($_GET);
            $this->file = file_get_contents("php://input");
            break;
        default:
            $this->_response('Invalid Method', 405);
            break;
        }
    }

	public function processAPI() {
        if ((int)method_exists($this, $this->module) > 0) {
            return $this->_response($this->{$this->module}($this->args));
        }
        return $this->_response("No module: $this->module", 404);
    }

    public function _response($data, $status = 200) {
		//header("HTTP/1.1 " . $status . " " . $this->_requestStatus($status));
		//return json_encode($data);
		return $data;
    }

    private function _cleanInputs($data) {
        $clean_input = Array();
        if (is_array($data)) {
            foreach ($data as $k => $v) {
                $clean_input[$k] = $this->_cleanInputs($v);
            }
        } else {
            $clean_input = trim(strip_tags($data));
        }
        return $clean_input;
    }

    private function _requestStatus($code) {
        $status = array(  
            200 => 'OK',
            404 => 'Not Found',   
            405 => 'Method Not Allowed',
            500 => 'Internal Server Error',
        ); 
        return ($status[$code])?$status[$code]:$status[500]; 
    }

	/**
	 * Generates and returns the XML
	 * 
	 * @param string $groupName The family of functions to access.
	 * @param string $methodName The specific operation to preform.
	 * @param array|null $params An optional array of parameters.
	 * @return string The returned XML as a string.
	 * @throws \InvalidArgumentException
	 */
	public function generateXML($params = null) {

		$xmlWriter = new XMLWriter();

		$xmlWriter->openMemory();

		$xmlWriter->startDocument('1.0', 'UTF-8');

		$xmlWriter->setIndent(4);

		/* Writes the result element. */
		$xmlWriter->startElement('result');

		if ($params !== null) {
			$xmlWriter->writeAttribute('null', 'false');
			
			foreach ($params as $key => $value) {
				if (is_array($value)) {
					foreach ($value as $item) {
						$xmlWriter->startElement($key);
						$xmlWriter->text($item);
						$xmlWriter->endElement();
					}
				} else {
					$xmlWriter->startElement($key);
					$xmlWriter->text($value);
					$xmlWriter->endElement();
				}
			}
		} else {
			$xmlWriter->writeAttribute('null', 'true');
		}

		/* Close the result element. */
		$xmlWriter->endElement();

		/* End the XML document. */
		$xmlWriter->endDocument();

		$xml = $xmlWriter->outputMemory(true);

		$file = fopen('./log/xml_request.xml', 'w');
		fwrite($file, $xml);
		fclose($file);
		
		return $xml;
	}
}